#!/usr/bin/env python3
import argparse, json, numpy as np
from pathlib import Path
from scipy.ndimage import gaussian_filter

def kernel_scale(path):
    K = np.load(path, allow_pickle=False).astype(float)  # (2,L,L)
    Kx, Ky = K
    # average incident links to a node map
    def sh(a,dx,dy): return np.roll(np.roll(a,dx,axis=1),dy,axis=0)
    node = (np.abs(Kx) + np.abs(sh(Kx,-1,0)) + np.abs(Ky) + np.abs(sh(Ky,0,-1))) / 4.0
    node = gaussian_filter(node, 1.0)

    L = node.shape[0]; cx=cy=(L-1)/2
    y,x = np.indices(node.shape)
    r = np.hypot(x-cx, y-cy).astype(int)
    sums = np.bincount(r.ravel(), weights=node.ravel())
    cnts = np.bincount(r.ravel())
    prof = sums/np.maximum(cnts,1)
    m = np.nanmax(prof)
    idx = np.argmax(prof <= 0.5*m)
    r_half = max(2, int(idx))
    return r_half, L

def recommend(rk, L):
    sigmas = [max(1, int(round(0.6*rk))), int(round(1.0*rk)), int(round(1.4*rk))]
    blur   = max(1, int(round(0.6*rk)))
    bmin   = max(12, int(round(0.05*L)))
    bmax   = int(round(0.25*L))
    return {"sigma_list": sigmas, "index_blur_sigma": blur, "lensing_b_min": bmin, "lensing_b_max": bmax}

def main():
    ap = argparse.ArgumentParser(description="Compute kernel-derived compactness scale r_k.")
    ap.add_argument("--kernel", required=True, help="Path to canonical kernel (2,L,L) .npy")
    ap.add_argument("--json", action="store_true", help="Emit JSON only.")
    args = ap.parse_args()

    rk, L = kernel_scale(args.kernel)
    rec = recommend(rk, L)
    out = {"kernel": args.kernel, "L": L, "r_k": rk, **rec}

    if args.json:
        print(json.dumps(out, indent=2))
    else:
        print(json.dumps(out, indent=2))
        print("\nYAML patch:")
        print(f"  sigma_list: {rec['sigma_list']}")
        print(f"  index_blur_sigma: {rec['index_blur_sigma']}")
        print(f"  lensing_b_min: {rec['lensing_b_min']}")
        print(f"  lensing_b_max: {rec['lensing_b_max']}")

if __name__ == "__main__":
    main()
